//
//  UIBackgroundTaskScheduler.swift
//  Do It
//
//  Created by Jim Dovey on 10/17/19.
//  Copyright © 2019 Jim Dovey. All rights reserved.
//

import UIKit
import Combine

struct UIBackgroundTaskScheduler<Target: Scheduler>: Scheduler {
    typealias SchedulerTimeType = Target.SchedulerTimeType
    typealias SchedulerOptions = Target.SchedulerOptions

    private let name: String?
    private var target: Target
    var now: SchedulerTimeType { target.now }
    var minimumTolerance: SchedulerTimeType.Stride { target.minimumTolerance }

    init(_ name: String? = nil, target: Target) {
        self.name = name
        self.target = target
    }

    func schedule(options: Self.SchedulerOptions?, _ action: @escaping () -> Void) {
        let taskID = UIApplication.shared.beginBackgroundTask(withName: self.name)
        target.schedule(options: options) {
            action()
            UIApplication.shared.endBackgroundTask(taskID)
        }
    }

    func schedule(after date: Self.SchedulerTimeType, tolerance: Self.SchedulerTimeType.Stride, options: Self.SchedulerOptions?, _ action: @escaping () -> Void) {
        let taskID = UIApplication.shared.beginBackgroundTask(withName: self.name)
        target.schedule(after: date, tolerance: tolerance, options: options) {
            action()
            UIApplication.shared.endBackgroundTask(taskID)
        }
    }

    private class CancellableBox {
        var cancellable: Cancellable? = nil
    }

    func schedule(after date: Self.SchedulerTimeType, interval: Self.SchedulerTimeType.Stride, tolerance: Self.SchedulerTimeType.Stride, options: Self.SchedulerOptions?, _ action: @escaping () -> Void) -> Cancellable {
        let box = CancellableBox()
        let taskID = UIApplication.shared.beginBackgroundTask(withName: name) {
            box.cancellable?.cancel()
            box.cancellable = nil
        }
        let canceller = target.schedule(after: date, interval: interval, tolerance: tolerance, options: options) {
            action()
            UIApplication.shared.endBackgroundTask(taskID)
        }
        box.cancellable = canceller
        return canceller
    }
}
